﻿/*  Author:     Ben Hall
 *  File:       PlayerCharacter.cs
 *  Date:       4/28/2011
 *  
 *  Purpose:    PlayerCharacter.cs extends the GameCharacter class and is used to represent
 *              the characters specifically used by the player.  The difference here is that
 *              only PlayerCharacters use items and equipment, and only PlayerCharacters can
 *              gain levels.
 *              
 *              This class holds methods used to handle level gain, as well as inventory
 *              and equipment manipulation.
 */

using System;
using System.Net;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Ink;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Animation;
using System.Windows.Shapes;

namespace OnlineGame
{
    public enum PlayerClass
    {
        Fighter,
        Rogue,
        Cleric,
        Wizard
    }

    public class PlayerCharacter : GameCharacter
    {
        public const int unarmedMinDamage = 1;
        public const int unarmedMaxDamage = 4;
        public const int experienceToLevel = 500;

        protected PlayerClass classType;
        protected GameMap currentMap;
        protected Position currentPosition;

        protected Armor equippedBody;
        protected Armor equippedShield;
        protected Weapon equippedWeapon;

        private int currentCurrency;
        private int currentExperience;
        //ITEMS//
        protected int[] itemInventory;
        protected int[] equipmentInventory;

        /*  This constructor is used only when initializing the game and should be replaced when the player
         *  either creates a new character or loads an existing character.
         */
        public PlayerCharacter()
        {
            this.characterName = "Blank";
            this.classType = PlayerClass.Fighter;
            this.level = 0;
            this.strength = new CharStat(1);
            this.stamina = new CharStat(1);
            this.agility = new CharStat(1);
            this.intelligence = new CharStat(1);
            this.faith = new CharStat(1);
            this.persona = new CharStat(1);
            this.maxHealth = DetermineMaxHealth();
            this.currentHealth = maxHealth;
            this.maxMagic = DetermineMaxMagic();
            this.currentMagic = maxMagic;

            this.currentCurrency = 0;
            this.currentExperience = 0;

            DetermineArmorValue();
            DetermineDamageValue();
        }

        /*  This constructor is used when creating a new PlayerCharacter, where level, experience and gold will
         *  always be a set value
         */
        public PlayerCharacter(String nameIn, PlayerClass classTypeIn, int strIn, int stmIn, int aglIn, int intIn, int fthIn, int prsIn) : this (nameIn, classTypeIn, 1, strIn, stmIn, aglIn, intIn, fthIn, prsIn, 0, 50)
        {
            
        }

        /*  This constructor is used when creating a PlayerCharacter object of an existing character, so it
         *  needs to include current experience and gold values.
         */
        public PlayerCharacter(String name, PlayerClass classTypeIn, int levelIn, int strIn, int stmIn, int aglIn, int intIn, int fthIn, int prsIn, int expIn, int currencyIn)
        {
            this.characterName = name;
            this.classType = classTypeIn;
            this.level = levelIn;
            
            this.strength = new CharStat(strIn);
            this.stamina = new CharStat(stmIn);
            this.agility = new CharStat(aglIn);
            this.intelligence = new CharStat(intIn);
            this.faith = new CharStat(fthIn);
            this.persona = new CharStat(prsIn);
            
            this.currentExperience = expIn;
            this.currentCurrency = currencyIn;

            this.maxHealth = DetermineMaxHealth();
            this.currentHealth = maxHealth;
            this.maxMagic = DetermineMaxMagic();
            this.currentMagic = maxMagic;

            this.defending = false;
            this.poisoned = false;
            this.covering = false;
            this.coveredBy = null;
            this.characterCovered = null;
            this.pDefenseChange = 0;
            this.mDefenseChange = 0;

            //this.skillsKnown = new bool[100];
            this.skillsKnown = new bool[MainPage.skillList.Length];
            for (int i = 0; i < skillsKnown.Length; i++)
            {
                skillsKnown[i] = false;
            }

            //this.itemInventory = new int[100];
            //this.equipmentInventory = new int[100];
            this.itemInventory = new int[MainPage.itemList.Length];
            this.equipmentInventory = new int[MainPage.equipmentList.Length];
            for (int i = 0; i < itemInventory.Length; i++)
            {
                itemInventory[i] = 0;
                equipmentInventory[i] = 0;
            }

            DetermineDamageValue();
            DetermineArmorValue();
        }

        //interface methods
        /*  GetCurrentMap() returns the player's current map
         */
        public GameMap GetCurrentMap()
        {
            return this.currentMap;
        }

        /*  SetCurrentMap() sets the player's current map to the given GameMap object
         */
        public void SetCurrentMap(GameMap mapIn)
        {
            this.currentMap = mapIn;
        }

        /*  GetCurrentPosition() return the player's current position on the map
         */
        public Position GetCurrentPosition()
        {
            return this.currentPosition;
        }

        /*  SetCurrentPosition() sets the player's current position on the map to the given Position value
         */
        public void SetCurrentPosition(Position posIn)
        {
            this.currentPosition = posIn;
        }

        /*  AddExperience() adds the given experience amount to the player's current experience
         */
        public void AddExperience(int totalExperienceReward)
        {
            this.currentExperience += totalExperienceReward;
        }

        /*  RemoveExperience() removes the given experience amount from the player's current experience.
         *  If the remaining experience value is negative, it is instead set equal to zero
         */
        public void RemoveExperience(int experienceRemoved)
        {
            if (currentExperience >= experienceRemoved)
            {
                this.currentExperience -= experienceRemoved;
            }
            else
            {
                this.currentExperience = 0;
            }
        }

        /*  GetCurrentExperience() returns the player's current experience
         */
        public int GetCurrentExperience()
        {
            return this.currentExperience;
        }

        /*  CheckForLevel() returns true if the player has enough experience to gain a level
         */
        public bool CheckForLevel()
        {
            if (currentExperience >= experienceToLevel)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /*  IncreaseLevel() increases the PlayerCharacter's level by one and removes experience from
         *  the character's current experience equal to the amount needed to level
         */
        public void IncreaseLevel()
        {
            this.level += 1;
            this.currentExperience -= experienceToLevel;
        }

        /*  LevelUp() takes in values for each stat and updates the PlayerCharacter's stats with the addition
         *  of these values to the previous values, then recalculates values such as maxHealth and maxMagic.
         */
        public void LevelUp(int strIncrease, int stmIncrease, int aglIncrease, int intIncrease, int fthIncrease, int prsIncrease)
        {
            this.strength = new CharStat(this.strength.stat + strIncrease);
            this.agility = new CharStat(this.agility.stat + aglIncrease);
            this.stamina = new CharStat(this.stamina.stat + stmIncrease);
            this.intelligence = new CharStat(this.intelligence.stat + intIncrease);
            this.faith = new CharStat(this.faith.stat + fthIncrease);
            this.persona = new CharStat(this.persona.stat + prsIncrease);
            //recalculate health and magic, and recover to full
            this.maxHealth = this.DetermineMaxHealth();
            this.currentHealth = this.maxHealth;
            this.maxMagic = this.DetermineMaxMagic();
            this.currentMagic = this.maxMagic;
        }

        /*  GetRanAway() returns the PlayerCharacter's ranAway boolean value
         */
        public bool GetRanAway()
        {
            return this.ranAway;
        }
        /*  SetRanAwaY() sets the PlayerCharacter's ranAway value to the given boolean
         */
        public void SetRanAway(bool ranAwayIn)
        {
            this.ranAway = ranAwayIn;
        }

        public int GetCurrentCurrencyAmount()
        {
            return this.currentCurrency;
        }
        public void AddCurrency(int currencyToAdd)
        {
            this.currentCurrency += currencyToAdd;
        }
        public void RemoveCurrency(int currencyToRemove)
        {
            this.currentCurrency -= currencyToRemove;
        }

                        /* EQUIPMENT HANDLING */
        /*  GetEquipmentInventory() returns the integer array showing the player's equipment inventory
         */
        public int[] GetEquipmentInventory()
        {
            return this.equipmentInventory;
        }

        /*  AddEquipment() takes in the index of an equipment in the equipmentInventory and the amount to add to
         *  the PlayerCharacter's inventory and adds that amount to its current inventory
         */
        public void AddEquipment(int equipIndex, int amount)
        {
            this.equipmentInventory[equipIndex] += amount;
        }

        /*  RemoveEquipment() takes in the index of an equipment in the equipmentInventory and removes one from that
         *  value.
         */
        public void RemoveEquipment(int equipIndex)
        {
            this.equipmentInventory[equipIndex]--;
        }

        /*  GetEquipmentQuantity() takes in the index of an equipment in equipmentInventory and returns the value
         *  in that array (equal to the number of that equipment that the PlayerCharacter possesses)
         */
        public int GetEquipmentQuantity(int equipIndex)
        {
            return this.equipmentInventory[equipIndex];
        }

        public int GetEquipmentQuantity(Equipment equipmentIn)
        {
            int quantity = GetEquipmentQuantity(this.GetEquipmentWithName(equipmentIn.GetItemName()));
            return quantity;
        }

        /*  HasEquipment() takes in the index of an equipment in equipmentInventory and returns true if the value
         *  in the array at that index is greater than zero.
         */
        public bool HasEquipment(int equipIndex)
        {
            if (this.equipmentInventory[equipIndex] > 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /*  HasEquipment() takes in the name of a piece of equipment and returns true if the PlayerCharacter
         *  possesses at least one piece of that equipment.
         */
        public bool HasEquipment(String equipNameIn)
        {
            return HasEquipment(GetEquipmentWithName(equipNameIn));
        }

        /*  GetEquipmentWithName() takes in the name of a piece of equipment and return the index in
         *  equipmentList where that piece of equipment is located.
         */
        public int GetEquipmentWithName(String equipNameIn)
        {
            for (int i = 0; i < MainPage.equipmentList.Length; i++)
            {
                if (MainPage.equipmentList[i] != null)
                {
                    if (MainPage.equipmentList[i].GetItemName().Equals(equipNameIn))
                    {
                        return i;
                    }
                }
            }
            return -1;
        }

        /*  GetEquippedWeapon() returns the PlayerCharacter's currently equipped weapon
         */
        public Weapon GetEquippedWeapon()
        {
            return this.equippedWeapon;
        }

        /*  SetEquippedWeapon() is used when loading a PlayerCharacter from a saved state.
         */
        public void SetEquippedWeapon(Weapon weaponIn)
        {
            this.equippedWeapon = weaponIn;
        }

        /*  GetEquippedBodyArmor() returns the PlayerCharacter's currently equipped body armor
         */
        public Armor GetEquippedBodyArmor()
        {
            return this.equippedBody;
        }

        /*  GetEquippedShield() returns the PlayerCharacter's currently equipped shield
         */
        public Armor GetEquippedShield()
        {
            return this.equippedShield;
        }

        /*  IsEquipped() takes in an Equipment object and returns true if that Equipment is one
         *  of the pieces currently equipped by the PlayerCharacter
         */
        public bool IsEquipped(Equipment equipmentIn)
        {
            if ((equipmentIn.Equals(this.equippedWeapon)) || (equipmentIn.Equals(this.equippedBody)) || (equipmentIn.Equals(this.equippedShield)))
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /*  Equip() takes in an Equipment object and sets the relevant Weapon or Armor variable equal to the
         *  given Equipment object.  It unequips pieces of Equipment as needed, such as removing equippedShield
         *  when the player equips a two-handed weapon
         *  
         *  The method then calls an update to display the new damage or armor values in CharacterGrid
         */
        public void Equip(Equipment equipIn)
        {
            if (equipIn is Weapon)
            {
                Weapon weaponToEquip = (Weapon)equipIn;
                //if weapon is two-handed, unequip shield first
                if (weaponToEquip.IsTwoHanded())
                {
                    Unequip(this.equippedShield);
                }
                this.equippedWeapon = weaponToEquip;
                DetermineDamageValue();
            }
            else  //equipIn is Armor
            {
                Armor armorToEquip = (Armor)equipIn;
                if (armorToEquip.GetArmorType() == ArmorType.Body)
                {
                    this.equippedBody = armorToEquip;
                }
                else if (armorToEquip.GetArmorType() == ArmorType.Shield)
                {
                    //if current weapon is two-handed, unequip it first
                    if (this.equippedWeapon.IsTwoHanded())
                    {
                        Unequip(this.equippedWeapon);
                    }
                    this.equippedShield = armorToEquip;
                }
                DetermineArmorValue();
            }
        }

        /*  Unequip() takes in a piece of equipment and sets the relevant Weapon or Armor value to
         *  null.  It then refreshes CharacterGrid to display the new damage or armor values.
         */
        public void Unequip(Equipment equipIn)
        {
            if (equipIn is Weapon)
            {
                this.equippedWeapon = null;
                DetermineDamageValue();
            }
            else  //equipIn is Armor
            {
                Armor unequip = (Armor)equipIn;
                if (unequip.GetArmorType() == ArmorType.Body)
                {
                    this.equippedBody = null;
                }
                else if (unequip.GetArmorType() == ArmorType.Shield)
                {
                    this.equippedShield = null;
                }
                DetermineArmorValue();
            }
        }

        /*  DetermineDamageValue() sets the PlayerCharacter's minDamage and maxDamage equal to
         *  the values of the equipped Weapon, or equal to unarmed min and max damage if no
         *  Weapon is equipped
         */
        private void DetermineDamageValue()
        {
            if (equippedWeapon != null)
            {
                this.minDamage = equippedWeapon.GetWeaponMinDamage();
                this.maxDamage = equippedWeapon.GetWeaponMaxDamage();
            }
            else
            {
                this.minDamage = unarmedMinDamage;
                this.maxDamage = unarmedMaxDamage;
            }
        }

        /*  DetermineArmorValue() sets the PlayerCharacter's armor value equal to the sum of
         *  the armor values of each equipped piece of armor.
         */
        private void DetermineArmorValue()
        {
            float totalArmorValue = 0;
            if (equippedBody != null)
            {
                totalArmorValue += equippedBody.GetProtectionValue();
            }
            if (equippedShield != null)
            {
                totalArmorValue += equippedShield.GetProtectionValue();
            }
            this.armor = totalArmorValue;
        }
        

                        /* ITEM HANDLING */
        /*  GetItemInventory() returns the integer array associated with the player's Item inventory
         */
        public int[] GetItemInventory()
        {
            return this.itemInventory;
        }

        /*  AddItem() takes in the index of an Item in itemInventory and increases the value by the given
         *  amount value.
         */
        public void AddItem(int itemIndex, int amount)
        {
            this.itemInventory[itemIndex] += amount;
        }

        /*  RemoveItem() takes in the index of an Item in itemInventory and decreases the value by one.
         */
        public void RemoveItem(int itemIndex)
        {
            this.itemInventory[itemIndex]--;
        }

        /*  GetItemQuantity() takes in the index of an Item in itemInventory and returns the value located
         *  there, representing the number of that Item the player possesses.
         */
        public int GetItemQuantity(int itemIndex)
        {
            return this.itemInventory[itemIndex];
        }

        public int GetItemQuantity(Item itemIn)
        {
            int quantity = GetItemQuantity(this.GetItemWithName(itemIn.GetItemName()));
            return quantity;
        }

        /*  HasItem() takes in the index of an Item in itemInventory and returns true is the value there
         *  is greater than zero.
         */
        public bool HasItem(int itemNum)
        {
            if (this.itemInventory[itemNum] > 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /*  HasItem() takes in the name of an Item and returns true if the player possesses at least one
         *  of that Item.
         */
        public bool HasItem(String itemNameIn)
        {
            return HasItem(GetItemWithName(itemNameIn));
        }

        /*  GetItemWithName() takes in the name of an Item and returns the index in itemList where that
         *  Item is located
         */
        public int GetItemWithName(String itemNameIn)
        {
            for (int i = 0; i < MainPage.itemList.Length; i++)
            {
                if (MainPage.itemList[i] != null)
                {
                    if (MainPage.itemList[i].GetItemName().Equals(itemNameIn))
                    {
                        return i;
                    }
                }
            }
            return -1;
        }

        public PlayerClass GetPlayerClass()
        {
            return this.classType;
        }

    }
}
